<?php

namespace Nidec\LaravelApiClient;

use Illuminate\Auth\Access\AuthorizationException;
use Illuminate\Auth\AuthenticationException;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Str;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\HttpKernel\Exception\UnprocessableEntityHttpException;
use Symfony\Component\Routing\Exception\NoConfigurationException;

trait CallsApi
{
    /**
     * @var array $cache Cache temporaire des ressources récupérées pour limiter les appels.
     */
    private static array $cache = [];
    private bool $forceFetch = false;
    const SERVICE_NAME = '';

    /**
     * Effectue un appel vers le service $service à l'endpoint $endpoint en utilisant la méthode $method.
     * ex: "GET https://service/resource/1"
     *
     * Le service doit être configuré dans config/services.
     *
     * @param string $service Nom du service tel que configuré dans config/services. (ex: "service")
     * @param string $endpoint Chemin vers la route à appeler sur le service. Commence par un "/". (ex: "/resource/1")
     * @param string $method Méthode/Verbe HTTP de la requête. (ex: "GET")
     * @param array $params Eventuels paramètres à passer à la requête.
     * @return array|null
     * @throws \Exception
     */
    protected function call(string $service, string $endpoint, string $method = 'GET', array $params = []): ?array
    {
        $url = config("services.$service.url");
        if (empty($url)) {
            throw new NoConfigurationException("Pas d'url configuré pour le service $service.");
        }

        // On log chaque appel avec le temps d'exécution de la requête pour aider le debug et
        // avoir un indicateur pour optimiser les appels
        Log::channel("api")->debug("Appel de l'endpoint $method $endpoint du service $service avec les paramètres : " . json_encode($params));
        $startTime = microtime(true);

        // TEMPORAIRE: On utilise des tokens d'accès personnels en attendant la validation de l'authentification normale.
        // On teste tous les tokens de l'utilisateur jusqu'à en trouver un qui fonctionne.
        foreach (\Auth::user()->personalAccessTokens as $token) {
            $response = Http::withHeaders([
                'Accept' => 'application/json',
                'Authorization' => "Bearer $token->token",
            ])->{strtolower($method)}($url . $endpoint, $params);
            if ($response->successful()) {
                break;
            }
        }
        Log::channel("api")->debug("Completed in " . microtime(true) - $startTime . " seconds");

        if (isset($response) && !$response->successful()) {
            Log::channel("api")->error("Api call returned with code {$response->status()}. Cause of error: " . json_encode($response->json()));
        }

        if (!isset($response) || $response->status() === 401) {
            throw new AuthenticationException("Aucun token valide trouvé pour s'authentifier au service $service.");
        } elseif ($response->status() === 403) {
            throw new AuthorizationException($response->json()['message'], 403);
        } elseif ($response->status() === 404) {
            throw new NotFoundHttpException($response->json()['message']);
        } elseif ($response->status() === 422) {
            throw new UnprocessableEntityHttpException($response->json()['message'], null, 422);
        } elseif (!$response->successful()) {
            throw new \Exception($response->json()['message'] ?? $response->body(), $response->status());
        }

        return $response->json();
    }

    /**
     * Cherche une resource dans le cache. Si elle n'est pas en cache,
     * exécute un appel vers le service, cache le résultat et le renvoie.
     *
     * @param string $resourceClass Classname de la resource à récupérer
     * @param int|null $id Id de la resource
     * @param array $params Paramètres à passer à la requête si la resource n'est pas dans le cache
     * @param array $meta Tableau de paramètres utilisés pour gérer des cas spécifiques
     * @return Resource|ResourceCollection|null
     */
    private function getResource(string $resourceClass, int $id = null, array $params = [], array $meta = []): Resource|ResourceCollection|null
    {
        $collection = empty($id);

        if (!$this->forceFetch) {
            $resource = $this->getFromCache($resourceClass, $id, $meta);
            if ($resource !== false) {
                return $collection ? $resourceClass::collection($resource) : new $resourceClass($resource);
            }
        }

        try {
            $resource = $this->call(
                self::SERVICE_NAME,
                $this->getUri($resourceClass, $id, $meta),
                'GET',
                $params
            );
        } catch (\Exception $e) {
            Session::flash('error', $e->getMessage());
            return $collection ? new ResourceCollection([]) : null;
        }
        $this->putInCache($resourceClass, $resource, $id);
        return $collection ? $resourceClass::collection($resource['data'] ?? $resource) : new $resourceClass($resource['data'] ?? $resource);
    }

    /**
     * Comme getResourceCollection mais renvoie un LengthAwarePaginator au lieu d'une ResourceCollection.
     *
     * @param string $resourceClass Classname de la resource à récupérer
     * @param array $params Paramètres à passer à la requête si la resource n'est pas dans le cache
     * @param array $meta Tableau de paramètres utilisés pour gérer des cas spécifiques
     * @return LengthAwarePaginator
     * @throws \Exception
     */
    public function getResourcePaginated(string $resourceClass, array $params = [], array $meta = []): LengthAwarePaginator
    {
        $data = $this->getResource($resourceClass, params: $params, meta: $meta);
        $path = 'meta.' . $this->getLocalPath($resourceClass);

        $meta = Arr::get(self::$cache, $path);

        return new LengthAwarePaginator($data, $meta['total'] ?? 0, $meta['perPage'] ?? 1, $meta['currentPage'] ?? null, $meta);
    }

    /**
     * Ajoute ou modifie une resource, met à jour le cache avec la resource créée ou modifiée et la renvoie.
     *
     * @param string $resourceClass Classname de la resource à créer ou modifier
     * @param int|null $id Id de la resource. Si null, il s'agit d'une création
     * @param array $params Paramètres à passer à la requête
     * @param array $meta Tableau de paramètres utilisés pour gérer des cas spécifiques
     * @param bool $collection
     * @return Resource|ResourceCollection|null
     */
    public function putResource(string $resourceClass, int $id = null, array $params = [], array $meta = [], bool $collection = false): Resource|ResourceCollection|null
    {
        try {
            $resource = $this->call(
                self::SERVICE_NAME,
                $this->getUri($resourceClass, $id, $meta),
                empty($id) || $collection ? 'POST' : 'PUT',
                $params
            );
        } catch (\Exception $e) {
            Session::flash('error', $e->getMessage());
            return $collection ? new ResourceCollection([]) : null;
        }
        $this->putInCache($resourceClass, $resource, $id);
        return $collection ? $resourceClass::collection($resource['data'] ?? $resource) : new $resourceClass($resource['data'] ?? $resource);
    }

    /**
     * Supprime une resource dans le service et dans le cache.
     *
     * @param string $resourceClass
     * @param int|null $id
     * @param array $params
     * @return bool
     */
    public function deleteResource(string $resourceClass, int $id = null, array $params = []): bool
    {
        $success = true;
        try {
            $this->call(
                self::SERVICE_NAME,
                $this->getUri($resourceClass, $id),
                "DELETE",
                $params,
            );
        } catch (\Exception $e) {
            Session::flash('error', $e->getMessage());
            $success = false;
        }
        // Si on supprime une collection de resources, on retire l'entièreté des resources de ce type du cache.
        // A voir si on veut optimiser pour retirer uniquement les resources passées en paramètre de la requête.
        $this->removeFromCache($resourceClass, $id);
        return $success;
    }

    /**
     * Recherche dans le cache une resource ou collection.
     * Revoie false si elle n'est pas trouvée dans le cache.
     *
     * @param string $resourceClass Classname de la resource à récupérer
     * @param int|null $id Identifiant de la resource. Si null, on cherche une collection
     * @param array $meta Array de paramètres utilisés pour gérer des cas spécifiques
     * @return array|false
     */
    private function getFromCache(string $resourceClass, int $id = null, array $meta = []): array|false
    {
        if (isset($meta['morph'])) {
            $type = $meta['morph'];
            $path = $this->getLocalPath($resourceClass);
            $resource = Arr::where(Arr::get(self::$cache, $path, []), function ($resource) use ($id, $type) {
                return $resource['external_id'] == $id && $resource['type'] === $type;
            });
            return empty($resource) ? false : $resource[array_key_first($resource)];
        }

        $path = $this->getLocalPath($resourceClass, $id);
        return Arr::get(self::$cache, $path, false);
    }

    /**
     * Stock la resource $data dans le cache.
     *
     * @param string $resourceClass Classname de la resource à stocker
     * @param array $data Resource ou collection à stocker en cache
     * @param int|null $id Id de la resource. Si null, on stock une collection
     * @return void
     */
    private function putInCache(string $resourceClass, array $data, int $id = null): void
    {
        $path = $this->getLocalPath($resourceClass, $id);

        if (isset($data['data'])) {
            Arr::set(self::$cache, "meta.$path", Arr::except($data, 'data'));
            $data = $data['data'];
        }

        if (!isset($data['id'])) {
            $data = array_combine(array_map(function ($item) {
                return $item['id'] ?? null;
            }, $data), $data);
        }

        Arr::set(self::$cache, $path, $data);
    }

    /**
     * Renvoie le chemin vers une resource dans le cache
     *
     * @param string $resourceClass
     * @param int|null $id
     * @return string
     */
    private function getLocalPath(string $resourceClass, int $id = null): string
    {
        $path = $this->baseResourceName($resourceClass);
        if ($id !== null) {
            $path .= ".$id";
        }
        return $path;
    }

    /**
     * Renvoie l'uri vers la resource au sein du service. Il est utilisé pour construire l'url
     * permettant de récupérer, créer, modifier ou supprimer une resource du service.
     * L'uri est déterminé selon une norme d'après le nom de la resource et l'id fourni.
     * On peut fournir un uri custom dans le paramètre $meta pour gérer les cas non-standards.
     *
     * @param string $resourceClass Classname de la resource
     * @param int|null $id Id de la resource. Si null, la cible est une collection
     * @param array $meta Array de paramètres utilisés pour gérer des cas spécifiques
     * @return string
     */
    private function getUri(string $resourceClass, int $id = null, array $meta = []): string
    {
        if (isset($meta['uri'])) {
            return $meta['uri'];
        }

        $uri = '/'.$this->baseResourceName($resourceClass);
        if (isset($meta['morph'])) {
            $uri .= "/{$meta['morph']}";
        }
        if ($id !== null) {
            $uri .= "/$id";
        }
        return $uri;
    }

    /**
     * Supprime la resource du cache à l'emplacement $path
     *
     * @param string $resourceClass Classname de la resource à supprimer
     * @param int|null $id Id de la resource
     * @return void
     */
    private function removeFromCache(string $resourceClass, ?int $id): void
    {
        $path = $this->getLocalPath($resourceClass, $id);
        Arr::forget(self::$cache, $path);
    }

    /**
     * Appel les fonctions de récupération des resources $resources afin de préremplir le cache.
     * Utilisé pour faire de l'eager-loading et optimiser les performances.
     * Force l'appel vers le service même si les resources existent en cache.
     * Peut être ainsi utilisé pour récupérer une collection filtrée en ignorant la collection en cache.
     *
     * @param string|array $resources Nom des resources à précharger. Une fonction "get$resource" doit exister.
     * Au lieu du nom de la resource, on peut passer un tableau de paramètres avec en clé le nom de la resource.
     * Ces paramètres sont passés à la fonction de récupération.
     * @return void
     */
    public function load(string|array $resources): void
    {
        $this->forceFetch = true;
        if (!is_array($resources)) {
            $resources = [$resources];
        }

        foreach ($resources as $key => $resource) {
            if (is_array($resource)) {
                $params = $resource;
                $resource = $key;
            }

            if (method_exists($this, "get" . ucfirst($resource))) {
                $this->{"get" . ucfirst($resource)}($params ?? null);
            }
        }
        $this->forceFetch = false;
    }

    private function baseResourceName(string $resourceClass): string
    {
        return Str::plural(strtolower(Str::kebab(last(explode('\\', $resourceClass)))));
    }
}
